% Interactive Explorer for multiwavelength absorption spectroscopy of a THREE
% component mixture. Compares quantitative measurement by weighted regression and
% TransFit methods.  Simulates photon noise and stray light and Izero shifts.

% Note: After executing this m-file, slide the Figure No. 1 and Figure No.
% 2 windows side-by-side so that they don't overlap.  

% You can adjust these (scalar) input variables in lines 41-45:
%    ArrayLength     Number of points in spectrum 
%    separation      Wavelength separation between the two components
%    width           Width of individual gaussian absorption bands within each spectrum 
%    InstWidth       Spectral Bandpas (resolution) of spectrometer 
%    absorbance[]    True absorbance of analytes [1,2,3]
%    noise           Fractional photon random noise on 100% intensity
%    IzeroShift      Random shift in 100% intensity (background absorption)
%    straylight      Fractional unabsorbed stray light (may be a constant or a
%                    spectrum of stray light values at each wavelength)
% List of Vectors:
% TSpectrum1,2,3 = true absorbance spectrum of component 1,2,3.
% ReferenceSpectrum1,2,3 = Broadened unit spectra of component 1,2,3.
% y = true transmission spectrum of mixture.
% InstFunction = instrument function 
% fa = Fourier transform of instrument function
% yobsd = noisy instrumentally broadened spectrum with noise, stray light
% f = frequency coordinate vector
% x = wavelength coordinate vector
% Functions called: Gaussian, broaden, fitMSpecConvThree (or other fitting function)
% Example: Just type TFit3 in the command window.
% Tom O'Haver (toh@umd.edu), August 2006.  Slider function by Matthew Jones.
clear
global x y z InstFunction linewidth absorbance InstWidth StrayLight 
global noise NoiseArray separation width IzeroShift 
close
figure(1);
clf
h=figure(1);
format compact

% Initial values of the user-adjustable parameters:
ArrayLength=100; % Number of points in the simulated arrays (spectra). Typically 50 - 1000
separation=ArrayLength/7; % Separation, in points, between the component spectra
width=ArrayLength/15;  % Width of the individual peaks within each spectrum
InstWidth=ArrayLength/10;   % Width of instrument function (spectral bandpass)
absorbance=[3 .1 5];  % Theoretical absorbances of the 3 components
noise=0.02;  % Random noise level when InstWidth = 1
IzeroShift=.01;  %  Random shift in 100% T intensity (background absorption)
StrayLight=0.01;  % Can be a scalar or a vector of length Arraylength

% Define frequency and wavelength coordinate vectors
x=[1:ArrayLength]';
TFit3Redraw(x,y,InstFunction,linewidth,absorbance,InstWidth,StrayLight,noise,NoiseArray,separation,width,IzeroShift)
h2=gca;  axis([.001 10 .001 10]);

% Draw the sliders and set the slider ranges, labels, initial positions, and
% associated functions called when moved
rtslid(h,@TFit3A1,h2,1,'Scale',[-2 1],'Def',log10(absorbance(1)),'Back',[1 1 1],'Label','A1(G)','Position',[0.03 0.7 0.03 0.22]);
rtslid(h,@TFit3A2,h2,1,'Scale',[-2 1],'Def',log10(absorbance(2)),'Back',[1 1 1],'Label','A2(B)','Position',[0.03 0.4 0.03 0.22]);
rtslid(h,@TFit3A3,h2,1,'Scale',[-2 1],'Def',log10(absorbance(3)),'Back',[1 1 1],'Label','A3(R)','Position',[0.03 0.1 0.03 0.22]);
rtslid(h,@TFit3separation,h2,0,'Scale',[0 ArrayLength/3],'Def',separation,'Back',[1 1 1],'Label','Sepn','Position',[0.95 0.7 0.03 0.22]);
rtslid(h,@TFit3InstWidth,h2,0,'Scale',[1 ArrayLength/5],'Def',InstWidth,'Back',[1 1 1],'Label','InstWidth','Position',[0.95 0.4 0.03 0.22]);
rtslid(h,@TFit3Noise,h2,0,'Scale',[0 .1],'Def',noise,'Back',[1 1 1],'Label','Noise','Position',[0.95 0.1 0.03 0.22]);